import 'package:flutter/material.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:wordpress_app/blocs/config_bloc.dart';
import 'package:wordpress_app/cards/card1.dart';
import 'package:wordpress_app/cards/card4.dart';
import 'package:wordpress_app/config/config.dart';
import 'package:wordpress_app/models/category.dart';
import 'package:wordpress_app/services/app_service.dart';
import 'package:wordpress_app/services/wordpress_service.dart';
import 'package:wordpress_app/utils/cached_image_with_dark.dart';
import 'package:wordpress_app/utils/empty_image.dart';
import 'package:wordpress_app/utils/loading_card.dart';
import 'package:wordpress_app/widgets/inline_ads.dart';
import 'package:wordpress_app/widgets/loading_indicator_widget.dart';
import '../models/article.dart';
import 'package:easy_localization/easy_localization.dart';

class CategoryBasedArticles extends ConsumerStatefulWidget {
  final Category category;
  final String? heroTag;
  const CategoryBasedArticles(
      {super.key, this.heroTag, required this.category});

  @override
  ConsumerState<CategoryBasedArticles> createState() =>
      _CategoryBasedArticlesState();
}

class _CategoryBasedArticlesState extends ConsumerState<CategoryBasedArticles> {
  final List<Article> _articles = [];
  ScrollController? _controller;
  int page = 1;
  bool? _loading;
  bool? _hasData;
  final int _postAmount = 10;

  @override
  void initState() {
    _controller =
        ScrollController(initialScrollOffset: 0.0, keepScrollOffset: true);
    _controller!.addListener(_scrollListener);
    _fetchArticles();
    _hasData = true;
    super.initState();
  }

  @override
  void dispose() {
    super.dispose();
    _controller!.dispose();
  }

  Future _fetchArticles() async {
    await WordPressService()
        .fetchPostsByCategoryId(widget.category.id, page, _postAmount)
        .then((value) {
      _articles.addAll(value);
      if (_articles.isEmpty) {
        _hasData = false;
      }
      setState(() {});
    });
  }

  Future<void> _scrollListener() async {
    var isEnd = AppService.isScrollNearEnd(_controller!);
    if (isEnd && _articles.isNotEmpty) {
      setState(() {
        page += 1;
        _loading = true;
      });
      await _fetchArticles().then((_) {
        setState(() {
          _loading = false;
        });
      });
    }
  }

  Future _onRefresh() async {
    setState(() {
      _loading = null;
      _hasData = true;
      _articles.clear();
      page = 1;
    });
    await _fetchArticles();
  }

  @override
  Widget build(BuildContext context) {
    final configs = ref.read(configBlocProvider).configs!;
    return Scaffold(
      body: RefreshIndicator(
        child: CustomScrollView(
          controller: _controller,
          slivers: <Widget>[
            SliverAppBar(
              automaticallyImplyLeading: true,
              leading: const BackButton(
                color: Colors.white,
              ),
              pinned: true,
              backgroundColor: Theme.of(context).primaryColor,
              expandedHeight: MediaQuery.of(context).size.height * 0.15,
              elevation: 0.5,
              flexibleSpace: _flexibleSpaceBar(),
            ),
            _hasData == false
                ? SliverFillRemaining(
                    child: Column(
                    children: [
                      SizedBox(
                        height: MediaQuery.of(context).size.height * 0.20,
                      ),
                      EmptyPageWithImage(
                          image: Config.noContentImage,
                          title: 'no-contents'.tr()),
                    ],
                  ))
                : SliverPadding(
                    padding: const EdgeInsets.all(20),
                    sliver: _articles.isEmpty && _hasData == true
                        ? SliverList.separated(
                            itemCount: 6,
                            separatorBuilder: (context, index) =>
                                const Divider(height: 40),
                            itemBuilder: (context, index) =>
                                const LoadingCard(height: 250),
                          )
                        : SliverList.separated(
                            itemCount: _articles.length,
                            separatorBuilder: (context, index) =>
                                const Divider(height: 40),
                            itemBuilder: (context, index) {
                              if ((index + 1) % configs.postIntervalCount ==
                                  0) {
                                return Column(
                                  children: [
                                    Card4(
                                      article: _articles[index],
                                      heroTag:
                                          'category_based_${widget.category.id}_${_articles[index].id}',
                                    ),
                                    const Divider(height: 60),
                                    const InlineAds(isSliver: true),
                                  ],
                                );
                              } else {
                                return Card1(
                                    article: _articles[index],
                                    heroTag:
                                        'category_based1_${widget.category.id}_${_articles[index].id}');
                              }
                            },
                          ),
                  ),
            if (_articles.isNotEmpty)
              SliverToBoxAdapter(
                child: Opacity(
                  opacity: _loading == true ? 1.0 : 0.0,
                  child: const LoadingIndicatorWidget(),
                ),
              ),
          ],
        ),
        onRefresh: () async => _onRefresh(),
      ),
    );
  }

  FlexibleSpaceBar _flexibleSpaceBar() {
    return FlexibleSpaceBar(
      centerTitle: false,
      background: Container(
        color: Theme.of(context).primaryColor,
        width: double.infinity,
        child: HeroMode(
          enabled: widget.heroTag != null &&
              widget.category.categoryThumbnail != null,
          child: Hero(
            tag: widget.heroTag ?? '',
            child: CustomCacheImageWithDarkFilterBottom(
                imageUrl: widget.category.categoryThumbnail, radius: 0.0),
          ),
        ),
      ),
      title: Text(
        widget.category.name.toString(),
        style: const TextStyle(color: Colors.white),
      ),
    );
  }
}
